package sidplay.audio;

import static libsidplay.common.SIDEndian.endian_little16;
import static libsidplay.common.SIDEndian.endian_little32;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;

import sidplay.Config;
import sidplay.audio.AudioConfig.encoding_t;

/**
 * @author Ken Hndel
 * 
 * Only unsigned 8-bit, and signed 16-bit, samples are supported. Endian-ess is
 * adjusted if necessary.
 * 
 * If number of sample bytes is given, this can speed up the process of closing
 * a huge file on slow storage media.
 * 
 */
public class WavFile extends AudioBase {

	/**
	 * @author Ken Hndel
	 * 
	 * little endian format
	 */
	class WavHeader {

		public static final int SIZE_OF = 44;

		public WavHeader() {
			// ASCII keywords are hex-ified.
			mainChunkID = new short[] { 0x52, 0x49, 0x46, 0x46 };
			length = new short[] { 0, 0, 0, 0 };
			chunkID = new short[] { 0x57, 0x41, 0x56, 0x45 };
			subChunkID = new short[] { 0x66, 0x6d, 0x74, 0x20 };
			subChunkLen = new short[] { 16, 0, 0, 0 };
			format = new short[] { 1, 0 };
			channels = new short[] { 0, 0 };
			sampleFreq = new short[] { 0, 0, 0, 0 };
			bytesPerSec = new short[] { 0, 0, 0, 0 };
			blockAlign = new short[] { 0, 0 };
			bitsPerSample = new short[] { 0, 0 };
			dataChunkID = new short[] { 0x64, 0x61, 0x74, 0x61 };
			dataChunkLen = new short[] { 0, 0, 0, 0 };
		}

		public byte[] getBytes() {
			return new byte[] { (byte) mainChunkID[0], (byte) mainChunkID[1],
					(byte) mainChunkID[2], (byte) mainChunkID[3],
					(byte) length[0], (byte) length[1], (byte) length[2],
					(byte) length[3], (byte) chunkID[0], (byte) chunkID[1],
					(byte) chunkID[2], (byte) chunkID[3], (byte) subChunkID[0],
					(byte) subChunkID[1], (byte) subChunkID[2],
					(byte) subChunkID[3], (byte) subChunkLen[0],
					(byte) subChunkLen[1], (byte) subChunkLen[2],
					(byte) subChunkLen[3], (byte) format[0], (byte) format[1],
					(byte) channels[0], (byte) channels[1],
					(byte) sampleFreq[0], (byte) sampleFreq[1],
					(byte) sampleFreq[2], (byte) sampleFreq[3],
					(byte) bytesPerSec[0], (byte) bytesPerSec[1],
					(byte) bytesPerSec[2], (byte) bytesPerSec[3],
					(byte) blockAlign[0], (byte) blockAlign[1],
					(byte) bitsPerSample[0], (byte) bitsPerSample[1],
					(byte) dataChunkID[0], (byte) dataChunkID[1],
					(byte) dataChunkID[2], (byte) dataChunkID[3],
					(byte) dataChunkLen[0], (byte) dataChunkLen[1],
					(byte) dataChunkLen[2], (byte) dataChunkLen[3], };
		}

		/**
		 * 'RIFF' (ASCII)
		 */
		short mainChunkID[] = new short[4];

		/**
		 * file length
		 */
		short length[] = new short[4];

		/**
		 * 'WAVE' (ASCII)
		 */
		short chunkID[] = new short[4];
		/**
		 * 'fmt ' (ASCII)
		 */
		short subChunkID[] = new short[4];
		/**
		 * length of subChunk, always 16 bytes
		 */
		short subChunkLen[] = new short[4];
		/**
		 * currently always = 1 = PCM-Code
		 */
		short format[] = new short[2];

		/**
		 * 1 = mono, 2 = stereo
		 */
		short channels[] = new short[2];
		/**
		 * sample-frequency
		 */
		short sampleFreq[] = new short[4];
		/**
		 * sampleFreq * blockAlign
		 */
		short bytesPerSec[] = new short[4];
		/**
		 * bytes per sample * channels
		 */
		short blockAlign[] = new short[2];

		short bitsPerSample[] = new short[2];

		/**
		 * keyword, begin of data chunk; = 'data' (ASCII)
		 */
		short dataChunkID[] = new short[4];

		/**
		 * length of data
		 */
		short dataChunkLen[] = new short[4];

	};

	long byteCount;

	final WavHeader defaultWavHdr = new WavHeader();

	WavHeader wavHdr = new WavHeader();

	RandomAccessFile file;

	/**
	 * whether file has been opened
	 */
	boolean isOpen;

	/**
	 * whether final header has been written
	 */
	boolean headerWritten;

	public WavFile() {
		isOpen = headerWritten = false;
	}

	@Override
	public byte[] open(AudioConfig cfg, String name) {
		return open(cfg, name, true);
	}

	byte[] open(AudioConfig cfg, final String name, final boolean overWrite) {
		long freq;
		int channels, bits;
		int blockAlign;
		int bufSize;

		bits = cfg.precision;
		channels = cfg.channels;
		freq = cfg.frequency;
		blockAlign = (bits >> 3) * channels;
		bufSize = (int) freq * blockAlign;
		cfg.bufSize = bufSize;

		// Setup Encoding
		cfg.encoding = encoding_t.AUDIO_SIGNED_PCM;
		if (bits == 8)
			cfg.encoding = encoding_t.AUDIO_UNSIGNED_PCM;

		if (name == null) {
			System.err.println("Audio(Wav): filename is null");
			System.exit(1);
			return null;
		}

		if (isOpen)
			close();

		byteCount = 0;

		// We need to make a buffer for the user
		_sampleBuffer = new byte /* uint_least8_t */[bufSize];

		// Fill in header with parameters and expected file size.
		endian_little32(wavHdr.length, 0, WavHeader.SIZE_OF - 8);
		endian_little16(wavHdr.channels, 0, channels);
		endian_little32(wavHdr.sampleFreq, 0, freq);
		endian_little32(wavHdr.bytesPerSec, 0, freq * blockAlign);
		endian_little16(wavHdr.blockAlign, 0, blockAlign);
		endian_little16(wavHdr.bitsPerSample, 0, bits);
		endian_little32(wavHdr.dataChunkLen, 0, 0);

		try {
			if (overWrite) {
				if (new File(name).exists()) {
					new File(name).delete();
				}
				file = new RandomAccessFile(name, "rw");
			} else {
				file = new RandomAccessFile(name, "rw");
				file.seek(file.length());
			}
			isOpen = true;
			System.out
					.println(String
							.format(
									"Audio(Wav): %d.0 Hz, %d bit, %d channel(s), %d bytes/s, block align: %d",
									freq, bits, channels, (int) freq
											* blockAlign, blockAlign));
		} catch (FileNotFoundException e) {
			isOpen = false;
			System.err.println("Audio(Wav): file not found! " + e);
			System.exit(1);
		} catch (IOException e) {
			isOpen = false;
			System.err.println("Audio(Wav): IO error! " + e);
			System.exit(1);
		}
		_settings = cfg;
		return _sampleBuffer;
	}

	/**
	 * After write call old buffer is invalid and you should use the new buffer
	 * provided instead.
	 */
	@Override
	public byte[] write() {
		try {
			if (isOpen) {
				int bytes = _settings.bufSize;
				if (!headerWritten) {
					file.write(wavHdr.getBytes());
					headerWritten = true;
				}

				byteCount += bytes;

				if (Config.WORDS_BIGENDIAN) {
					if (_settings.precision == 16) {
						byte[] /* int_least8_t * */pBuffer = (byte[] /* int_least8_t * */) _sampleBuffer;
						for (int /* uint_least32_t */n = 0; n < _settings.bufSize; n += 2) {
							byte tmp = pBuffer[n + 0];

							pBuffer[n + 0] = pBuffer[n + 1];
							pBuffer[n + 1] = tmp;
						}
					}
				}
				file.write((byte[] /* char* */) _sampleBuffer, 0, bytes);
			}
		} catch (IOException e) {
			return null;
		}
		return _sampleBuffer;
	}

	@Override
	public void pause() {
		// TODO Auto-generated method stub
	}

	@Override
	public void close() {
		try {
			if (isOpen) {
				endian_little32(wavHdr.length, 0, byteCount + WavHeader.SIZE_OF
						- 8);
				endian_little32(wavHdr.dataChunkLen, 0, byteCount);
				file.seek(0);
				file.write(wavHdr.getBytes(), 0, WavHeader.SIZE_OF);
				file.close();
				isOpen = false;
				_sampleBuffer = null;
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@Override
	public void reset() {
	}

	@Override
	public String extension() {
		return ".wav";
	}

}
